/**
* \file: am_api_socket_client.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdbool.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/inotify.h>
#include <limits.h>
#include <errno.h>
#include <fcntl.h>

#include "am_api_socket_client.h"
#include "am_api_dispatcher.h"
#include "am_api_fsm.h"

#include "utils/logger.h"
#include "utils/global_constants.h"
#include "ipc/message_recvr.h"
#include "ipc/message_sendr.h"

#undef SUN_LEN
#define SUN_LEN(su) ((sizeof(*(su)) - sizeof((su)->sun_path)) + strlen((su)->sun_path))

static int connection_socket = -1;

static int inotify_fd = -1;
static int base_dir_wd = -1;
static int automounter_dir_wd = -1;
static message_queue_t sender_queue;
static message_buffer_t *receiver_buffer;
//------------------------------------- private attributes ------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member declaration ----------------------------------------------------
static int am_api_socket_client_create_socket(void);
static error_code_t am_api_socket_client_do_connect(int socket_to_connect);
static void am_api_socket_client_on_socket_event(uint32_t events);
static void am_api_socket_client_on_ready_to_send(void);
static void am_api_socket_client_on_data_present_event(void);
static void am_api_socket_client_on_inotify_event(uint32_t events);

static void am_api_socket_client_on_base_dir_event(uint32_t mask);
static void am_api_socket_client_on_automounter_dir_event(void);
static void am_api_socketclient_check_automounter_dir_present(void);
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member definition ------------------------------------------------------
error_code_t am_api_socket_client_init(void)
{
	error_code_t result=RESULT_OK;

	if (receiver_buffer!=NULL || connection_socket!=-1)
		am_api_socket_client_deinit();

	receiver_buffer=message_recvr_create_receiver_buffer();
	if (receiver_buffer==NULL)
		result=RESULT_NORESOURCE;

	if (result==RESULT_OK)
		result=message_sendr_init();

	return result;
}

void am_api_socket_client_deinit(void)
{
	am_api_socket_client_disconnect();

	if (receiver_buffer!=NULL)
	{
		message_recvr_destroy_receiver_buffer(receiver_buffer);
		receiver_buffer=NULL;
	}

	message_sendr_deinit();
}

error_code_t am_api_socket_client_connect(void)
{
	error_code_t result;

	//not initialized
	if (receiver_buffer==NULL)
		return RESULT_INVALID;

	if (connection_socket != -1)
		am_api_socket_client_disconnect();

	result=am_api_socket_client_check_daemon_running();
	if (result==RESULT_DAEMON_NOT_RUNNING)
		logger_log_error("The automounter daemon is not running.");
	else if (result==RESULT_ACCESS_RIGHTS)
		logger_log_error("The application has not the access rights to connect with the automounter daemon.");

	if (result!=RESULT_OK)
		return result;

	//daemon running, we are trying to connect with it
	connection_socket=am_api_socket_client_create_socket();
	if (connection_socket==-1)
		return RESULT_SOCKET_ERR;

	//we got a socket, connect with it
	result=am_api_dispatcher_register_event_fd(connection_socket, am_api_socket_client_on_socket_event, EPOLLIN);

	if (result==RESULT_OK)
		result=am_api_socket_client_do_connect(connection_socket);

	if (result!=RESULT_OK)
		am_api_socket_client_disconnect();
	else
	{
		//init the sender queue
		message_sendr_init_message_queue(&sender_queue);
		//mark receiver buffer as being empty
		message_buffer_mark_empty(receiver_buffer);
	}

	return result;
}

void am_api_socket_client_disconnect(void)
{
	if (connection_socket!=-1 && receiver_buffer!=NULL)
	{
		am_api_dispatcher_deregister_event_fd(connection_socket);
		close(connection_socket);
		connection_socket=-1;
		message_sendr_empty_queue(&sender_queue);
	}
}

error_code_t am_api_socket_client_send_message(message_buffer_t *msg_buffer)
{
	error_code_t result;

	if (connection_socket==-1 || receiver_buffer==NULL)
		return RESULT_INVALID;
	result=message_sendr_send_buffer(msg_buffer,connection_socket);
	if (result==RESULT_MSG_PARTLY_PROCESSED)
	{
		message_sendr_add_buffer_to_send_queue(msg_buffer,&sender_queue);
		result=am_api_dispatcher_change_eventfd_eventmask(connection_socket, am_api_socket_client_on_socket_event,
				EPOLLIN|EPOLLOUT);
	}
	else
		message_sendr_putback_buffer(msg_buffer);

	if (result==RESULT_SOCKET_ERR)
		am_api_fsm_signal_connection_lost();

	return result;
}

error_code_t am_api_socket_client_check_daemon_running(void)
{
	struct stat st;
	//check if the automouter base dir is present

	logger_log_debug("AUTOMOUNTER_API - Checking if the automounter is running.");
	if (stat(AUTOMOUNTER_RUN_DIR, &st)!=0)
	{
		logger_log_debug("AUTOMOUNTER_API - Automounter run directory not found -> Daemon not running.");
		return RESULT_DAEMON_NOT_RUNNING;
	}

	if (stat(AUTOMOUNTER_SOCKET_READY_FILE,&st)==0)
	{
		logger_log_debug("AUTOMOUNTER_API - The daemon is running.");
		return RESULT_OK;
	}

	if (errno==ENOENT)
	{
		logger_log_debug("AUTOMOUNTER_API - Automounter ready file not found -> Daemon not running.");
		return RESULT_DAEMON_NOT_RUNNING;
	}

	logger_log_debug("AUTOMOUNTER_API - Access rights problems detected.");

	return RESULT_ACCESS_RIGHTS;
}

error_code_t am_api_socket_client_start_waiting_for_socket(void)
{
	//not initialized
	if (receiver_buffer==NULL)
		return RESULT_INVALID;

	//someone forget to close us ...
	if (inotify_fd!=-1)
		am_api_socket_client_stop_waiting_for_socket();

	inotify_fd=inotify_init();
	if (inotify_fd==-1)
		return RESULT_NORESOURCE;

	base_dir_wd=inotify_add_watch(inotify_fd,BASE_RUN_DIR,IN_CREATE|IN_DELETE);
	if (base_dir_wd==-1)
		return RESULT_NORESOURCE;

	am_api_dispatcher_register_event_fd(inotify_fd,am_api_socket_client_on_inotify_event,EPOLLIN);
	//from now on we receive events

	//emulate an event that is normally fired in case of changes in the base run directory (/run). This checks whether
	//the automounter is already running and sets up the respective inotify watches.
	am_api_socket_client_on_base_dir_event(IN_ISDIR);

	return RESULT_OK;
}

void am_api_socket_client_stop_waiting_for_socket(void)
{
	if (inotify_fd!=-1)
	{
		am_api_dispatcher_deregister_event_fd(inotify_fd);
		close(inotify_fd);
		//the close call here destroys all watches as well. We need to reset the descriptors here
		//because we are using them for state evaluation
		inotify_fd=-1;
		base_dir_wd=-1;
		automounter_dir_wd=-1;
	}
}
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member definition -----------------------------------------------------
static int am_api_socket_client_create_socket(void)
{
	int new_socket = socket(AF_UNIX, SOCK_STREAM, 0);

	if (new_socket < 0)
	{
		logger_log_error("Unable to create a unix socket.");
		return -1;
	}

	return new_socket;
}

static error_code_t am_api_socket_client_do_connect(int socket_to_connect)
{
	struct sockaddr_un serveraddr;
	memset(&serveraddr, 0, sizeof(serveraddr));
	serveraddr.sun_family = AF_UNIX;
	strcpy(serveraddr.sun_path, AUTOMOUNTER_CTRL_SOCKET);

	if (connect(socket_to_connect, (struct sockaddr *)&serveraddr, SUN_LEN(&serveraddr)) < 0)
	{
		if (errno==EACCES)
		{
			logger_log_error("The application has not the access rights to connect with the automounter daemon.");
			return RESULT_ACCESS_RIGHTS;
		}
		else
		{
			logger_log_error("Error connecting with the automounter at socket: %s",AUTOMOUNTER_CTRL_SOCKET);
			return RESULT_CONNECT_TO_DAEMON_ERR;
		}
	}

	//make it nonblocking
	if (fcntl(socket_to_connect,F_SETFL,O_NONBLOCK)!=0)
	{
		logger_log_error("Unable to set the new socket to non blocking mode. Closing the connection.");
		close(socket_to_connect);
		return RESULT_SOCKET_ERR;
	}

	return RESULT_OK;
}

static void am_api_socket_client_on_socket_event(uint32_t events)
{
	if ((events & EPOLLIN)!=0)
		am_api_socket_client_on_data_present_event();
	if ((events & EPOLLOUT)!=0)
		am_api_socket_client_on_ready_to_send();
}

static void am_api_socket_client_on_ready_to_send(void)
{
	error_code_t result;
	logger_log_debug("AUTOMOUNTER_API - Got an event that the socket is now able to receive"
			" data again. Sending queued messages now.");

	result=message_sendr_send_queued_buffers(&sender_queue,connection_socket);
	if (result != RESULT_MSG_PARTLY_PROCESSED)
		// we either managed freeing our queue or we run into a sending error. In both cases we can
		// remove the pollout event from our epoll
		am_api_dispatcher_change_eventfd_eventmask(connection_socket,am_api_socket_client_on_socket_event,EPOLLIN);
	else
		logger_log_debug("AUTOMOUNTER_API - Socket buffer full. Buffers remained in the queue. Sending them later.");

	if (result == RESULT_SOCKET_ERR)
	{
		logger_log_error("AUTOMOUNTER_API - Connection lost to automounter while processing messages.");
		am_api_fsm_signal_connection_lost();
	}
}

static void am_api_socket_client_on_data_present_event(void)
{
	error_code_t result;
	result=message_recvr_receive_msg(receiver_buffer,connection_socket);

	if (result==RESULT_SOCKET_CLOSED || result==RESULT_SOCKET_ERR)
		am_api_fsm_signal_connection_lost();
	else if (result==RESULT_SOCKET_MSG_CORRUPT)
		logger_log_error("AUTOMOUNTER_API - Received a corrupt message from the automounter. Ignoring it!!");
	else if (result==RESULT_MSG_PARTLY_PROCESSED)
		logger_log_error("AUTOMOUNTER_API - Received parts of a message from the automounter. Waiting for the rest.");
	else if (result==RESULT_INVALID)
		logger_log_error("AUTOMOUNTER_API - We tried to read data from a connection without having initialized the "
				"generic message module before. Implementation error!!");
	else if (result==RESULT_NORESOURCE)
		logger_log_error("AUTOMOUNTER_API - We are running into resource issues. Can't do anything from here.");
	else
		am_api_dispatcher_on_message_received(receiver_buffer);

	if (result!=RESULT_MSG_PARTLY_PROCESSED)
	{
		//We can mark the buffer as being empty here safely for following reasons
		// - In case the message has been received completely, it has been already dispatched
		// - In case we got a corrupt message or we ran into some sort of connection error, we are not
		//		interested in the content of the buffer any more
		message_buffer_mark_empty(receiver_buffer);
	}
}

static void am_api_socket_client_on_inotify_event(uint32_t events)
{
	char buffer[sizeof(struct inotify_event) + NAME_MAX + 1];
	struct inotify_event *event=(struct inotify_event *)buffer;

	//we only registered for EPOLLIN
	(void)events;

	if (read(inotify_fd,event,sizeof(struct inotify_event) + NAME_MAX + 1)<=0)
	{
		logger_log_error("AUTOMOUNTER_API - We had some problems reading out the inofity event even"
				" though we got woken up because of available data.");
		return;
	}

	if (event->wd==base_dir_wd)
		am_api_socket_client_on_base_dir_event(event->mask);
	else if (event->wd==automounter_dir_wd)
		am_api_socket_client_on_automounter_dir_event();
}

static void am_api_socket_client_on_base_dir_event(uint32_t mask)
{
	// we are waiting for a directory
	if ((mask & IN_ISDIR) == 0)
		return;

	// ok someone either created or removed a directory, check if our directory is there or not and react respectively
	am_api_socketclient_check_automounter_dir_present();
}

static void am_api_socket_client_on_automounter_dir_event(void)
{
	//something happened in our directory, check if the socket_rdy file appeared
	if (am_api_socket_client_check_daemon_running()==RESULT_OK)
		am_api_fsm_signal_automounter_appeared();
}

static void am_api_socketclient_check_automounter_dir_present(void)
{
	struct stat st;
	bool am_dir_present;
	am_dir_present=(stat(AUTOMOUNTER_RUN_DIR,&st)==0);
	//we found our dir but no one is listing. Add a watch for listening and check if the file is already there now
	if (am_dir_present && automounter_dir_wd==-1)
	{
		// add the watch
		automounter_dir_wd=inotify_add_watch(inotify_fd,AUTOMOUNTER_RUN_DIR,IN_CREATE);
		if (automounter_dir_wd==-1)
			logger_log_error("AUTOMOUNTER_API - Problems adding a watch to inotify while waiting"
					" for the automounter to appear.");

		// check if the socket_rdy file is already there
		am_api_socket_client_on_automounter_dir_event();
	}

	if (!am_dir_present && automounter_dir_wd!=-1)
	{
		//someone remove the directory but we still have a watch on it, remove the watch
		//This could happen if the automounter comes up and goes down very fast again.
		inotify_rm_watch(inotify_fd, automounter_dir_wd);
		automounter_dir_wd=-1;
	}
}
//---------------------------------------------------------------------------------------------------------------------
